<?php
/**
 * Elementor Widget
 * @package foresty
 * @since 1.0.0
 */

namespace Elementor;

class Foresty_Team_List_Widget extends Widget_Base
{

    /**
     * Get widget name.
     *
     * Retrieve Elementor widget name.
     *
     * @return string Widget name.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_name()
    {
        return 'foresty-team-list-widget';
    }

    /**
     * Get widget title.
     *
     * Retrieve Elementor widget title.
     *
     * @return string Widget title.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_title()
    {
        return esc_html__('Team List', 'foresty-core');
    }

    /**
     * Get widget icon.
     *
     * Retrieve Elementor widget icon.
     *
     * @return string Widget icon.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_icon()
    {
        return 'eicon-person';
    }

    /**
     * Get widget categories.
     *
     * Retrieve the list of categories the Elementor widget belongs to.
     *
     * @return array Widget categories.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_categories()
    {
        return ['foresty_widgets'];
    }

    /**
     * Register Elementor widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function register_controls()
    {

        $this->start_controls_section(
            'slider_settings_section',
            [
                'label' => esc_html__('Team Settings', 'foresty-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
        $this->add_control('team_grid', [
            'label' => esc_html__('Team Grid', 'foresty-core'),
            'type' => Controls_Manager::SELECT,
            'options' => array(
                'col-lg-2' => esc_html__('col-lg-2', 'foresty-core'),
                'col-lg-3' => esc_html__('col-lg-3', 'foresty-core'),
                'col-lg-4' => esc_html__('col-lg-4', 'foresty-core'),
                'col-lg-6' => esc_html__('col-lg-6', 'foresty-core'),
            ),
            'default' => 'col-lg-3',
            'description' => esc_html__('Select Case Study Grid', 'foresty-core')
        ]);
        $this->add_control(
            'pagination',
            [
                'label' => esc_html__('Pagination', 'foresty-core'),
                'type' => Controls_Manager::SWITCHER,
                'description' => esc_html__('you can set yes to show pagination.', 'foresty-core'),
                'default' => 'yes'
            ]
        );
        $this->add_control(
            'pagination_alignment',
            [
                'label' => esc_html__('Pagination Alignment', 'foresty-core'),
                'type' => Controls_Manager::SELECT,
                'options' => array(
                    'left' => esc_html__('Left Align', 'foresty-core'),
                    'center' => esc_html__('Center Align', 'foresty-core'),
                    'right' => esc_html__('Right Align', 'foresty-core'),
                ),
                'description' => esc_html__('you can set pagination alignment.', 'foresty-core'),
                'default' => 'left',
                'condition' => array('pagination' => 'yes')
            ]
        );
        $this->add_control('total', [
            'label' => esc_html__('Total Posts', 'foresty-core'),
            'type' => Controls_Manager::TEXT,
            'default' => '-1',
            'description' => esc_html__('enter how many course you want in masonry , enter -1 for unlimited course.')
        ]);
        $this->add_control('category', [
            'label' => esc_html__('Category', 'foresty-core'),
            'type' => Controls_Manager::SELECT2,
            'multiple' => true,
            'options' => foresty_core()->get_terms_names('team-cat', 'id'),
            'description' => esc_html__('select category as you want, leave it blank for all category', 'foresty-core'),
        ]);
        $this->add_control('order', [
            'label' => esc_html__('Order', 'foresty-core'),
            'type' => Controls_Manager::SELECT,
            'options' => array(
                'ASC' => esc_html__('Ascending', 'foresty-core'),
                'DESC' => esc_html__('Descending', 'foresty-core'),
            ),
            'default' => 'ASC',
            'description' => esc_html__('select order', 'foresty-core')
        ]);
        $this->add_control('orderby', [
            'label' => esc_html__('OrderBy', 'foresty-core'),
            'type' => Controls_Manager::SELECT,
            'options' => array(
                'ID' => esc_html__('ID', 'foresty-core'),
                'title' => esc_html__('Title', 'foresty-core'),
                'date' => esc_html__('Date', 'foresty-core'),
                'rand' => esc_html__('Random', 'foresty-core'),
                'comment_count' => esc_html__('Most Comments', 'foresty-core'),
            ),
            'default' => 'ID',
            'description' => esc_html__('select order', 'foresty-core')
        ]);
        $this->end_controls_section();

        $this->start_controls_section(
            'styling_settings_section',
            [
                'label' => esc_html__('Team style Settings', 'foresty-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        $this->add_control(
            'item_padding',
            [
                'label' => esc_html__('Padding', 'foresty-core'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .team__item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        $this->add_control('title_color', [
            'label' => esc_html__('Title Color', 'foresty-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .team__item h4 a" => "color: {{VALUE}}"
            ]
        ]);
        $this->add_control('designation_color', [
            'label' => esc_html__('designation Color', 'foresty-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .team__item span" => "color: {{VALUE}}"
            ]
        ]);
		$this->add_control('social_hover_bg_color', [
            'label' => esc_html__('social hover bg Color', 'foresty-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .team__item-image-icon a:hover" => "background-color: {{VALUE}}"
            ]
        ]);
		$this->add_control('social_hover_border_color', [
            'label' => esc_html__('social hover border Color', 'foresty-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
				"{{WRAPPER}} .team__item-image-icon a:hover" => "border-color: {{VALUE}}",
            ]
        ]);
		$this->add_control('hover_bg_color', [
            'label' => esc_html__('hover bg Color', 'foresty-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .team__item::after" => "background-color: {{VALUE}}"
            ]
        ]);
		$this->add_control('designation_hover_color', [
            'label' => esc_html__('designation hover Color', 'foresty-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .team__item:hover span" => "color: {{VALUE}}"
            ]
        ]);
        $this->end_controls_section();

    }


    /**
     * Render Elementor widget output on the frontend.
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();
        //query settings
        $total_posts = $settings['total'];
        $category = $settings['category'];
        $order = $settings['order'];
        $orderby = $settings['orderby'];
        //other settings
        $pagination = $settings['pagination'] ? false : true;
        $pagination_alignment = $settings['pagination_alignment'];
        //setup query
        $args = array(
            'post_type' => 'team',
            'posts_per_page' => $total_posts,
            'order' => $order,
            'orderby' => $orderby,
            'post_status' => 'publish'
        );

        if (!empty($category)) {
            $args['tax_query'] = array(
                array(
                    'taxonomy' => 'team-cat',
                    'field' => 'term_id',
                    'terms' => $category
                )
            );
        }
        $post_data = new \WP_Query($args);
        ?>

        <div class="row justify-content-center">
            <?php
            while ($post_data->have_posts()) : $post_data->the_post();
                $post_id = get_the_ID();
                $img_id = get_post_thumbnail_id($post_id) ? get_post_thumbnail_id($post_id) : false;
                $img_url_val = $img_id ? wp_get_attachment_image_src($img_id, 'foresty_classic_team', false) : '';
                $img_url = is_array($img_url_val) && !empty($img_url_val) ? $img_url_val[0] : '';
                $img_alt = get_post_meta($img_id, '_wp_attachment_image_alt', true);

                $team_single_meta_data = get_post_meta(get_the_ID(), 'foresty_team_options', true);
                $social_icons = isset($team_single_meta_data['social-icons']) && !empty($team_single_meta_data['social-icons']) ? $team_single_meta_data['social-icons'] : '';
                $designation = isset($team_single_meta_data['designation']) && !empty($team_single_meta_data['designation']) ? $team_single_meta_data['designation'] : '';

                ?>
                <div class="<?php echo esc_attr($settings['team_grid']); ?> col-md-6 wow fadeInUp" data-wow-duration="1.2s" data-wow-delay=".2s">
                    <div class="team__item mb-4">
                        <div class="team__item-image">
                            <img src="<?php echo esc_url($img_url)?>" alt="<?php echo $img_alt?>">
                            <div class="team__item-image-icon social-icon">
                                <?php
                                    if (!empty($social_icons)) {
                                        foreach ($social_icons as $item) {
                                            printf('<a href="%1$s"><i class="%2$s"></i></a>', $item['url'], $item['icon']);
                                        }
                                    }
                                ?>
                            </div>
                        </div>
                        <h4><a href="<?php the_permalink() ?>"><?php echo get_the_title(); ?></a></h4>
                        <span><?php echo $designation; ?></span>
                    </div>
                </div>
            <?php endwhile; ?>
            
            <div class="col-lg-12">
                <div class="blog-pagination text-<?php echo esc_attr($pagination_alignment) ?> margin-top-20">
                    <?php
                    if (!$pagination) {
                        Foresty()->post_pagination($post_data);
                    }
                    ?>
                </div>
            </div>
        </div>
        <?php
    }
}

Plugin::instance()->widgets_manager->register_widget_type(new Foresty_Team_List_Widget());